function exportBody(body, spreadsheet_name)

spreadsheet_name = spreadsheet_name + ".xlsx";

% Main data
mainData = table('Size', [0 3], ...
    'VariableTypes', ["string", "cell", "string"], ...
    'VariableNames', ["Main Parameters", "value", "unit"]);

mainData(end+1, :) = {"Full name", {body.fullName}, ""};
mainData(end+1, :) = {"Class", {body.class}, ""};

% Masses
if isfield(body, "curbMass")
    mainData(end+1, :) = {"Curb mass", {body.curbMass}, "kg"};
end
if isfield(body, "chassisMass")
    mainData(end+1, :) = {"Chassis mass", {body.chassisMass}, "kg"};
end
if isfield(body, "maxLoad")
    mainData(end+1, :) = {"Max load mass", {body.maxLoad}, "kg"};
end
if isfield(body, "grossMass")
    mainData(end+1, :) = {"Gross mass", {body.grossMass}, "kg"};
end

% Resistant forces
mainData(end+1, :) = {"Resistant forces method", {body.resForceMethod}, ""};
switch body.resForceMethod
    case "roadload"

        mainData(end+1, :) = {"F0 (coast-down coefficient)", {body.f0}, "N"};
        mainData(end+1, :) = {"F1 (coast-down coefficient)", {body.f1}, "N / (m/s)"};
        mainData(end+1, :) = {"F2 (coast-down coefficient)", {body.f2}, "N / (m/s)^2"};

        % Axles
        mainData(end+1, :) = {"Axle specification", {body.axleMethod}, ""};
        mainData(end+1, :) = {"Tyre", {body.tyre}, ""};
        mainData(end+1, :) = {"Axle inertia", {body.axles.inertia(1)}, ""};

        if strcmp(body.axleMethod, "explicit")
            error("Resistant forces method ""roadload"" is incompatible with Axle specification ""explicit""")
        end

    case "analytic"

        if isfield(body, 'airDragCoef') && body.airDragCoef ~= 0
            mainData(end+1, :) = {"Air drag coefficient", {body.airDragCoef}, ""};
        end
        if isfield(body, 'frontArea') && body.frontArea ~= 0
            mainData(end+1, :) = {"Front area", {body.frontArea}, "m^2"};
        end
        mainData(end+1, :) = {"CdA", {body.CdA}, "m^2"};
        
        % Axles
        mainData(end+1, :) = {"Axle specification", {body.axleMethod}, ""};

        switch body.axleMethod
            case "concentrated"
                % Explicit RRC coefficients
                mainData(end+1, :) = {"Rolling resistance coeff. RRC0", {body.rollResCoeff0}, "N"};
                mainData(end+1, :) = {"Rolling resistance coeff. RRC1", {body.rollResCoeff1}, "N/(m/s)"};
                mainData(end+1, :) = {"Tyre", {body.tyre}, ""};
                mainData(end+1, :) = {"Axle inertia", {body.axles.inertia(1)}, ""};

            case "explicit"
                body.axles = structfun(@(x) x(:), body.axles, 'UniformOutput', false);
                axles = struct2table(body.axles);
                writetable(axles, spreadsheet_name, 'Sheet', 'axles')

            otherwise
                error("The axle specification method must be ""explicit"" or ""concentrated""");
                
        end

    otherwise
        error("The Resistant forces method must be ""roadload"" or ""anaylitic""");

end

% Write to spreadsheet
writetable(mainData, spreadsheet_name, Sheet = "main")

end


